/*
 * $Id: dictint.c,v 1.3 2004/11/06 18:20:09 mark Exp $
 *
 * NAME
 *      dictint - dictionary interface routines
 *
 * SYNOPSIS
 *      int df_set_type(int type);
 *          Sets type of dictionary to use - either BTREE or UNIXDICT
 *      int df_open(char *filename);
 *          Opens dictionary file named by filename
 *      int df_position(char *key, int pos);
 *          Position dictionary at key (BTREE) or pos (UNIXDICT)
 *      int df_next(char *s, int *pos, int *more);
 *          Return next word from dictionary in s.  If UNIXDICT, current
 *          file position is returned via pos.  more is set false if
 *          no more words exist
 *      int df_close()
 *          Closes open dictionary file
 *
 * DESCRIPTION
 *      Dictint provides the necessary routines to allow maf to operate
 *      against either a btree index file or a UNIX dictionary as the
 *      dictionary file.
 *
 * MODIFICATION HISTORY
 * Mnemonic    Date    Rel Who
 * DICTINT     01Oct05 1.0 MPW
 *     Written.
 * DICTINT     04Jun06 1.1 MPW
 *     Revised to support BT 2.0 API
 *
 * Copyright (C) 2001, 2004 Mark Willson.
 *
 * This file is part of the maf program.
 *
 * The maf program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * The maf program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <stdio.h>
#include <string.h>
#include "dictint.h"
#include "ds.h"

#ifdef USE_BTREE
#include "btree.h"
#endif

static int df_type = -1;
static int df_init = FALSE;
static FILE *df_file = NULL;

#ifdef USE_BTREE
static BTA* bt_file;
#endif

int df_set_type(int type)
{
#ifdef USE_BTREE
    if (type == BTREE || type == UNIXDICT) {
#else
    if (type == UNIXDICT) {
#endif
        df_type = type;
        return(TRUE);
    }
    return(FALSE);
}

int df_open(char *filename)
{
    if (df_type == BTREE) {
#ifdef USE_BTREE
        if (!df_init) {
            btinit();
            df_init = TRUE;
        }
        bt_file = btopn(filename,0,0);
        return((bt_file != NULL)?0:1);
#endif
    }
    else {
        // dictionary is sorted ignoring case
        df_file = opendict(filename,TRUE);
        return((df_file != NULL)?0:1);
    }
}

int df_position(char *key,int pos)
{
    int val,
        status;
    char buff[WORDSIZE+1];

    if (df_type == BTREE) {
#ifdef USE_BTREE
        // Construct anagdict format word (i.e. prefixed signature)
        strsort(key,buff);
        strcat(buff,PREFIXDELIMSTR);
        strcat(buff,key);
        status = bfndky(bt_file,buff,&val);
        return((status == QNOKEY || status == 0)?0:1);
#endif
    }
    else {
        if (setposition(pos)) {
            if (strlen(key) != 0) {
                /* not a position before first entry; discard found
                   entry */
                getentry(buff);
            }
            return(0);
        }
        return(1);
    }
}

int df_next(char *s, int *pos, int *more)
{
    int val,
        status;
    char buff[WORDSIZE+1];
    char *p;

    if (df_type == BTREE) {
#ifdef USE_BTREE
        *more = TRUE;
        status = bnxtky(bt_file,buff,&val);
        if (status == QNOKEY) {
            *more = FALSE;
        }
        else if (status != 0) {
            return(-2);
        }

        p = strchr(buff,PREFIXDELIMCHAR);
        if (p == NULL) return(-2);
        strcpy(s,p+1);
        return(0);
#endif
    }
    else {
        *pos = getentry(s);
        *more = (*pos >= 0);
        if (*pos < 0) *pos = 0; /* make end of file into rewind */
        return(0);
    }
}

int df_close()
{
    if (df_type == BTREE) {
#ifdef USE_BTREE
        return(btcls(bt_file));
#endif
    }
    else {
        return(fclose(df_file));
    }
}
