"""
NAME

  catalog.py - Provides class for reading local DTD etc catalog files

SYNOPSIS

  Class catalog.Catalog()
  Object build(filename)
  Object showDict()
  Object resolveEntity(entity_name)

DESCRIPTION

  Provides support for XML entity resolution via local catalog files.  Uses
  shlex to handle lexical analysis of catalog files.
 
  Create an object using Catalog, and invoke the build method, passing
  the name of the root catalog file.  If build() returns True, a
  in-memory catalog (possibly empty) has been built.  Use the
  resolveEntity() method to return the local file name of an entity
  reference.  If resolved, a string is returned, otherwise None.

NOTES

  Written from eyeballing the versions of catalog files I have locally.
  Therefore, may not work in other circumstances.
  
  Probably won't work for nested CATALOG entries.

MODIFICATION HISTORY
  Mnemonic   Rel   Date     Who
  catalog    1.0   20050922 mpw
     Created.

$Id: catalog.py,v 1.1 2014/04/18 19:45:45 mark Exp $
"""
import sys
import getopt
import os
import shlex

class Lex(shlex.shlex):
    def sourcehook(self,filename):
        # make filename local to parent catalog directory
        fn = self.dir+"/"+filename.strip('"')
        f = open(fn)
        return (fn,f)

class Catalog:
    def __init__(self):
        self.cat_dict = dict()
        self.lex = None

    def build(self,filename):
        try:
            f = open(filename)
        except IOError:
            return False
        
        self.lex = Lex(f)
        self.lex.source = "CATALOG"     # keyword for sub catalogs
        self.lex.dir = os.path.dirname(filename)
        self.lex.commenters = "-"       # comment character (maybe --)
        tok = "start"
        while len(tok) != 0:
            if tok == "PUBLIC":
                id = self.lex.get_token().strip('"')
                self.cat_dict[id] = os.path.dirname(self.lex.infile)+"/"+\
                               self.lex.get_token().strip('"')
            tok = self.lex.get_token()
        return True
    
    def showDict(self):
        for e,f in self.cat_dict.iteritems():
            print "%s ->\n\t %s" % (e,f)

    def resolveEntity(self,ent):
        if self.cat_dict.has_key(ent):
            return self.cat_dict[ent]
        else:
            return None

if __name__ == '__main__':
    cat = Catalog()
    ref = "ISO 8879:1986//ENTITIES Publishing//EN//XML"
    if cat.build("/home/mark/dtd/CATALOG"):
        print "Dump of Catalog dictonary:"
        cat.showDict()
        print "Attempting to resolve:", ref
        print cat.resolveEntity(ref)
    else:
        print "Unable to build in-memory CATALOG"
